//
//  TWLoginViewController.m
//  Twitch Plugin
//
//  Created by Auston Stewart on 7/25/14.
//  Copyright (c) 2014 Twitch Interactive, Inc. All rights reserved.
//

#import "TWLoginViewController.h"
#import "TWHUDView.h"
#import "NSURLAdditions.h"
#import "TwitchKit.h"

static NSString *const TWBroadcastScopeList = @"user_read channel_read channel_editor channel_commercial sdk_broadcast metadata_events_edit chat_login";

@implementation TWLoginViewController {
	BOOL _loggedInSuccessfully; // returning NO from webView:shouldStartLoadWithRequest:navigationType: causes webView:didFailLoadWithError: to be called
	NSString *_authToken;
	NSString *_error;
	NSString *_redirectURI;
}

- (id)initWithClientID:(NSString *)clientID redirectURI:(NSString *)redirectURI mode:(TWLoginViewControllerMode)mode
{
	if ((self = [super initWithURL:[NSURL URLWithString:[NSString stringWithFormat:@"https://api.twitch.tv/kraken/oauth2/authorize?response_type=token&login_type=%@&client_id=%@&redirect_uri=%@&scope=%@",mode == TWLoginViewControllerModeSignup ? @"signup" : @"login", clientID, redirectURI, [TWBroadcastScopeList tw_stringByPercentEncodingString]]]])) {
		
		_redirectURI = redirectURI;
	}
	
	return self;
}

- (void)dismissModalLoginViewController
{
	[self.presentingViewController dismissViewControllerAnimated:YES completion:NULL];
	
	NSMutableDictionary *userInfo = [[NSMutableDictionary alloc] initWithCapacity:2];
	if (_error.length) userInfo[TKErrorUserInfoKey] = _error;
	if (_authToken.length) userInfo[TKAuthTokenUserInfoKey] = _authToken;
	
	[[NSNotificationCenter defaultCenter] postNotificationName:TKLoginViewWasDismissedNotification object:[TwitchKit sharedPlugin] userInfo:userInfo];
}

#pragma mark -

- (BOOL) webView:(UIWebView *) webView shouldStartLoadWithRequest:(NSURLRequest *) request navigationType:(UIWebViewNavigationType) navigationType {
	
	if ([[request.URL absoluteString] hasPrefix:_redirectURI]) {
	
		NSDictionary *authenticationInformation = request.URL.tw_fragmentDictionary; // The API is documented to give us this..
		if (!authenticationInformation.count)
			authenticationInformation = request.URL.tw_queryDictionary; // ..but, to be safe, also check for this.
		if (!authenticationInformation.count) {
			_error = @"Unable to parse response from authentication endpoint.";
			return NO;
		}
		
		if ([authenticationInformation objectForKey:@"error"]) {
		
			_error = [authenticationInformation objectForKey:@"error"];
			_loggedInSuccessfully = NO;
		}
		else {
			_loggedInSuccessfully = YES;
		}
				
		_authToken = authenticationInformation[@"access_token"];
				
		[self dismissModalLoginViewController];
		
		return NO;
	}
	
	return YES;
}

- (void) webView:(UIWebView *) webView didFailLoadWithError:(NSError *) error {
	if (![error.userInfo[NSURLErrorFailingURLStringErrorKey] hasPrefix:_redirectURI] &&
		![error.userInfo[NSURLErrorFailingURLStringErrorKey] isEqual:webView.request.URL.absoluteString]) {
		[TWHUDView showErrorHUD];
    }
}

@end

@implementation UINavigationController (LoginViewControllerPresentation)

+ (UINavigationController *) loginNavigationControllerForClientID:(NSString *)clientID redirectURI:(NSString *)redirectURI mode:(TWLoginViewControllerMode)mode {
	
	// Clear cookies for .twitch.tv to prevent Accept-Language from being overridden
	for (NSHTTPCookie *cookie in [[[NSHTTPCookieStorage sharedHTTPCookieStorage] cookies] copy]) {
		
		if ([[cookie domain] hasSuffix:@"twitch.tv"]) {
			
			[[NSHTTPCookieStorage sharedHTTPCookieStorage] deleteCookie:cookie];
		}
	}
	
	TWLoginViewController *loginViewController = [[TWLoginViewController alloc] initWithClientID:clientID redirectURI:redirectURI mode:mode];
	loginViewController.title = mode == TWLoginViewControllerModeLogin ? NSLocalizedString(@"Log In",@"Log In view title") : NSLocalizedString(@"Sign Up",@"Sign Up view title");
	
	UINavigationController *navigationController = [[UINavigationController alloc] initWithRootViewController:loginViewController];
	
	loginViewController.navigationItem.leftBarButtonItem = [[UIBarButtonItem alloc] initWithTitle:NSLocalizedString(@"Cancel", @"Cancel button title") style:UIBarButtonItemStylePlain target:loginViewController action:@selector(dismissModalLoginViewController)];
	
	return navigationController;
}

@end
